/*************************************************************************
 * The contents of this file are subject to the MYRICOM MYRINET          *
 * EXPRESS (MX) NETWORKING SOFTWARE AND DOCUMENTATION LICENSE (the       *
 * "License"); User may not use this file except in compliance with the  *
 * License.  The full text of the License can found in LICENSE.TXT       *
 *                                                                       *
 * Software distributed under the License is distributed on an "AS IS"   *
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied.  See  *
 * the License for the specific language governing rights and            *
 * limitations under the License.                                        *
 *                                                                       *
 * Copyright 2003 - 2004 by Myricom, Inc.  All rights reserved.          *
 *************************************************************************/

static const char __idstring[] = "@(#)$Id: mx_irecv.c,v 1.122 2006/09/19 19:47:08 bgoglin Exp $";
 
#include "mx_auto_config.h"
#include "myriexpress.h"
#include "mx__lib_types.h"
#include "mx_pin.h"
#include "mx__request.h"
#include "mx__segment.h"
#  include "mx__partner.h"
#include "mx__endpoint.h"
#include "mx__self.h"
#include "mx_util.h"
#include "mx__lib.h"
#include "mx__error.h"
#include "mx__shmem.h"
#include "mx__ack.h"

MX_FUNC(mx_return_t)
#ifndef MX_KERNEL
mx_irecv(mx_endpoint_t ep,
	 mx_segment_t *segments_list, uint32_t segments_count,
	 uint64_t match_info, uint64_t match_mask,
	 void *context, mx_request_t *request)
#else
mx_kirecv(mx_endpoint_t ep,
	 mx_segment_t *segments_list, uint32_t segments_count,
	 mx_pin_type_t pin_type,
	 uint64_t match_info, uint64_t match_mask,
	 void *context, mx_request_t *request)
#endif
{
  union mx_request *p;
  union mx_request *unexp;
  uint32_t l;
  mx_return_t ret_val;
#ifndef MX_KERNEL
  mx_pin_type_t pin_type = MX_PIN_UNDEFINED;
#endif
  uint32_t ctxid = CTXID_FROM_MATCHING(ep, match_info);    

  if (segments_count > MX_MAX_SEGMENTS) {
    ret_val = MX_BAD_SEG_CNT;
    goto abort_without_lock;
  }

  if (match_info & ~match_mask) {
    ret_val = MX_BAD_MATCH_MASK;
    goto abort_without_lock;
  }

  /* check that there's no wildcard in the context id range */
  if (!CHECK_MATCHING_WITH_CTXID(ep, match_mask)) {
    ret_val = MX_BAD_MATCHING_FOR_CONTEXT_ID_MASK;
    goto abort_without_lock;
  }

  ret_val = MX_SUCCESS;
  MX__MUTEX_LOCK(&ep->lock);

  p = mx__rl_alloc(ep);
  if (p == NULL) {
    ret_val = MX_NO_RESOURCES;
    goto abort_with_lock;
  }

  MX__CACHE_SEGMENTS(p->recv, segments_list, segments_count);
  if ((p->recv.count > 1) && (p->recv.segments == NULL)) {
    ret_val = MX_NO_RESOURCES;
    goto abort_with_request;
  }
  p->recv.memory_context = pin_type;
  if (MX_DEBUG_CSUM && mx__opt.csum) {
    match_mask &= UINT64_C(0x0000ffffffffffff);
  }
#if MX_ONE_SIDED
  match_mask |= MX__ONESIDED_MASK;
#endif
  
  p->recv.match_info = match_info;
  p->recv.match_mask = match_mask;
  p->recv.basic.status.context = context;
#if MX_DEBUG
  p->recv.basic.status.xfer_length = -1;
#endif
  p->recv.basic.mcp_handle = -1;
  p->recv.r_length = mx__total_segment_length(p->recv.segments,
					      p->recv.count);
  p->recv.basic.state = MX__REQUEST_STATE_PENDING;
  p->recv.basic.type = MX__REQUEST_TYPE_RECV;
  p->recv.basic.partner = NULL;
  p->recv.basic.wq = NULL;
  p->recv.basic.acquired_by_wait_any = 0;
  p->recv.unexpected = 0;
  p->recv.put_target = 0;

  MX__EP_STATS_INC(ep, irecv);

  unexp = mx__endpoint_match_unexpected(ep, match_info, match_mask);
  if (unexp == NULL) {
    mx__enqueue_request(&ep->ctxid[ctxid].recv_reqq, p);
  } else {
    void *unexp_ptr = MX_SEGMENT_PTR_TO_VA(unexp->recv.segment.segment_ptr);
    mx_assert((unexp->recv.basic.state & (MX__REQUEST_STATE_COMPLETED|MX__REQUEST_STATE_PENDING))
	      && unexp->recv.basic.status.code == MX_STATUS_SUCCESS);
    p->basic.status.code = unexp->basic.status.code;
    p->basic.status.source = unexp->basic.status.source;
    p->basic.status.match_info = unexp->basic.status.match_info;
    p->basic.status.msg_length = unexp->basic.status.msg_length;
    p->basic.state |= MX__REQUEST_STATE_RECV_MATCHED;
#ifndef MX_KERNEL
    if (unexp->recv.basic.type == MX__REQUEST_TYPE_RECV_SELF) {
      if (unexp->recv.self_send_req) {
	ep->sendself_count -= 1;
      }
      mx__self_copy(ep, unexp->recv.self_send_req, p, unexp_ptr);

    } else
#endif
#if MX_USE_SHMEM
    if (unexp->recv.basic.type == MX__REQUEST_TYPE_RECV_SHM) {
      p->basic.status.xfer_length =
	MX_MIN(p->recv.r_length, p->recv.basic.status.msg_length);
      mx__shm_copy(ep, unexp->recv.shm_peer_endpt, unexp->recv.shm_peer_req,
		   &unexp->recv.shm_src_segs, unexp->recv.shm_src_nsegs,
		   unexp->recv.shm_src_session, p);
      mx__recv_complete(ep, p, MX_STATUS_SUCCESS);

    } else
#endif
    if (unexp->recv.basic.type != MX__REQUEST_TYPE_RECV_LARGE) {
      l = mx__copy_to_segments(p->recv.segments, p->recv.count,
			       pin_type,
			       0, unexp_ptr,
			       unexp->recv.segments->segment_length);
      p->basic.status.xfer_length = MX_MIN(unexp->basic.status.msg_length,
					   p->recv.r_length);
      if (unexp->recv.basic.state & MX__REQUEST_STATE_COMPLETED) {
	mx__recv_complete(ep, p, unexp->recv.basic.status.code);
      } else {
	p->recv.msg_seq = unexp->recv.msg_seq;
	p->recv.accum = unexp->recv.accum;
	p->recv.r_mask = unexp->recv.r_mask;
	
	/* Replace unexp in partial list with real recv. */
	mx__replace_partner_request(&unexp->basic.partner->partialq, unexp, p);
	p->basic.partner = unexp->basic.partner;
	/* This partially received request is pending. */
	mx__enqueue_request(&ep->multifrag_recvq, p);
      }
    } else {
      /* rndv message */
      p->recv.msg_seq = unexp->recv.msg_seq;
      p->recv.r_mask = unexp->recv.r_mask;
      p->basic.partner = unexp->basic.partner;
      p->recv.unexpected = 0;
      p->recv.remote_rdma = unexp->recv.remote_rdma;
      p->basic.status.xfer_length = MX_MIN(unexp->basic.status.msg_length,
					   p->recv.r_length);
      mx_assert (unexp->recv.basic.state & MX__REQUEST_STATE_PENDING);
      mx__queue_large_recv(ep, p);
    }
    mx__spliceout_request(&ep->ctxid[ctxid].unexpq, unexp);
    if (unexp_ptr) {
      mx_free(unexp_ptr);
      ep->unexp_queue_length -= p->recv.basic.status.msg_length;
      ep->ordered_unexp_length -= unexp->recv.ordered_unexp_weight;
      mx__process_early(ep, unexp->basic.partner);
      mx__liback(ep, unexp->basic.partner);
      mx_assert((int32_t)ep->unexp_queue_length >= 0);
      mx_assert((int32_t)ep->ordered_unexp_length >= 0);
    }
    mx__rl_free(ep, unexp);
  }
  mx__luigi(ep);

  *request = p;
  MX__MUTEX_UNLOCK(&ep->lock);
  return ret_val;

 abort_with_request:
  mx__rl_free(ep, p);
 abort_with_lock:
  MX__MUTEX_UNLOCK(&ep->lock);
 abort_without_lock:
  return ret_val;
}
